%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This program solves the life-cycle model, based on work by 
% Gertler 1999 model with government
% This script simulates the model to calculate the impact of exogenous
% changes in fiscal policy (four variables: g, b, e and h, which are govt 
% spending, debt, social security and health spending respectively) 
% and demographics and TFP on R*. 

% The script calls gov_initial_ss.mod and gov_simulation.mod which are two 
% Dynare model scripts. 

% The main output is the matrix containing the simulations of the interest
% rate, called R_sim_inc, which needs to be pasted to Excel spreadsheet
% Figures 2 - 10 - 15.xlsx, tab: Full simulation

% Lukasz Rachel, London, 19 September 2018
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all;
close all;

% read in data for simulation

% fiscal exogenous variables
fiscal_sim_data = csvread('gov_sim_data.csv',1,0); %fiscal ratios: debt, G, and social security and healthcare

% demographics and tfp
demog_sim_data = csvread('demog_sim_data.csv',1,0); % read in data for TFP, population growth, gamma (prob of dying) and omega (prob of retiring)

% drop years
year            = demog_sim_data(:,5);
fiscal_sim_data = fiscal_sim_data(:,1:4);
demog_sim_data  = demog_sim_data(:,1:4);

% make them equal length by repeating (repmat) the last value (it's the new
% steady state). always extend the shorter series
T_fiscal = numel(fiscal_sim_data(:,1));
T_demog  = numel(demog_sim_data(:,1));
if T_fiscal < T_demog
    fiscal_sim_data = [fiscal_sim_data; repmat(fiscal_sim_data(T_fiscal,:),T_demog-T_fiscal, 1)];
elseif T_fiscal > T_demog
    demog_sim_data  = [demog_sim_data; repmat(demog_sim_data(T_demog,:),T_fiscal-T_demog, 1)];
else     
end 

% Merge data together
data = [fiscal_sim_data, demog_sim_data];
T    = numel(data(:,1));

% How much of the simulation to keep
T_sim_end = 100;
year_sim = 1971:1:1971+99;

% split the data into separate variables
bbar_data   = data(:,1);
gbar_data   = data(:,2);
ebar_data   = data(:,3);
hbar_data   = data(:,4);
x_data      = data(:,5);
n_data      = data(:,6);
ggamma_data = data(:,7);
oomega_data = data(:,8);

% Initial SS
% use 1971 values for initial SS
bbar_init_ss     = bbar_data(1);
gbar_init_ss     = gbar_data(1);
ebar_init_ss     = ebar_data(1);
hbar_init_ss     = hbar_data(1);
x_init_ss        = x_data(1);
n_init_ss        = n_data(1);
ggamma_init_ss   = ggamma_data(1);
oomega_init_ss   = oomega_data(1);

% Final SS
bbar_fin_ss      = bbar_data(T);
gbar_fin_ss      = gbar_data(T);
ebar_fin_ss      = ebar_data(T);
hbar_fin_ss      = hbar_data(T);
x_fin_ss         = x_data(T);
n_fin_ss         = n_data(T);
ggamma_fin_ss    = ggamma_data(T);
oomega_fin_ss    = oomega_data(T);

% Parameter values
aalpha=0.667; %labor share
bbeta=0.98;   %discount factor
ddelta=0.1;   %depreciation
ssigma=0.5;   %IES
ggamma_ss=0.9;%Probability of dying as a retiree
n_ss=1.01;    %Population growth (gross)

save my_parameters.mat aalpha bbeta ddelta ssigma ggamma_ss n_ss

% run the initial steady state. 

% note that the numbers in the oo_.steady_state will be different than in
% the paper because in the paper we divide by output. 

dynare gov_initial_ss.mod

% run the model feeding in all exogenous variables

dynare gov_simulation.mod noclearall
R_all = oo_.endo_simul(2,1:T_sim_end);
R_all = R_all';

%% Rerun the model feeding exogenous variables one by one

% Preliminaries
R_sim = zeros(T_sim_end,numel(data(1,:)));
data_loop = repmat(data(1,:),T,1); %start with no exogenous shocks (constant exo)
number_exo = numel(data(1,:));

for option = 1:2

% START LOOP OVER SHOCKS
for i=1:number_exo
    
    data_loop(:,i)=data(:,i); %fill in one column at a time
    
% generate exo variables paths for dynare
bbar_data(:) = data_loop(:,1);
gbar_data(:) = data_loop(:,2);
ebar_data(:) = data_loop(:,3);
hbar_data(:) = data_loop(:,4);
x_data(:)    = data_loop(:,5);
n_data(:)    = data_loop(:,6);
ggamma_data  = data_loop(:,7);
oomega_data  = data_loop(:,8);

% generate final steady state values
bbar_fin_ss  = data_loop(T,1);
gbar_fin_ss  = data_loop(T,2);
ebar_fin_ss  = data_loop(T,3);
hbar_fin_ss  = data_loop(T,4);
x_fin_ss     = data_loop(T,5);
n_fin_ss     = data_loop(T,6);
ggamma_fin_ss= data_loop(T,7);
oomega_fin_ss= data_loop(T,8);

% Options: (1) feed shocks one by one; (2) feed shocks cumulatively
% This matters because of non-linearities will be captured in the new
% steady state; so the effects are not additive when fed cumulatively
% Both options need to be solved for and



if option == 1;
    
    dynare gov_simulation.mod noclearall
    R_sim(:,i) = oo_.endo_simul(2,1:T_sim_end)'; %store R
    data_loop = repmat(data(1,:),T,1); %restore clean 'no shock matrix'
    
else 
    
    dynare gov_simulation.mod noclearall
    R_sim(:,i) = oo_.endo_simul(2,1:T_sim_end)' - oo_.endo_simul(2,1); %store R (in changes)
    if i>1 %convert into marginal impact of the next shock
        R_sim(:,i) = R_sim(:,i) - sum(R_sim(:,1:i-1),2);
    end 
        
end 

end 

%% Calculate the change in rates from initial SS

R_all_inc     = R_all(:) - R_all(1);
R_sim_first   = repmat(R_sim(1,:),numel(R_sim(:,1)),1);
R_sim_inc(option,:,:)     = R_sim - R_sim_first;

end 

% Collect Results:
RESULTS_1 = squeeze(R_sim_inc(1,:,:));
RESULTS_2 = squeeze(R_sim_inc(2,:,:));










